<?php
// change-password.php
session_start();

// اگر کاربر لاگین نکرده باشد، به صفحه ورود هدایت شود
if (!isset($_SESSION['user_id'])) {
    header('Location: auth.php');
    exit;
}

require_once __DIR__ . '/config/database.php';

$user_id = $_SESSION['user_id'];
$message = '';
$success = false;

// دریافت اطلاعات کاربر
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = :id");
$stmt->execute(['id' => $user_id]);
$user = $stmt->fetch();

if (!$user) {
    session_destroy();
    header('Location: auth.php');
    exit;
}

// پردازش فرم تغییر رمز عبور
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_password'])) {
    $current_password = $_POST['current_password'] ?? '';
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';
    
    // اعتبارسنجی
    $errors = [];
    
    if (empty($current_password)) {
        $errors[] = 'رمز عبور فعلی الزامی است.';
    }
    
    if (empty($new_password)) {
        $errors[] = 'رمز عبور جدید الزامی است.';
    } elseif (strlen($new_password) < 6) {
        $errors[] = 'رمز عبور جدید باید حداقل 6 کاراکتر باشد.';
    }
    
    if (empty($confirm_password)) {
        $errors[] = 'تکرار رمز عبور جدید الزامی است.';
    } elseif ($new_password !== $confirm_password) {
        $errors[] = 'رمز عبور جدید و تکرار آن مطابقت ندارند.';
    }
    
    // بررسی رمز عبور فعلی
    if (!empty($current_password) && !password_verify($current_password, $user['password'])) {
        $errors[] = 'رمز عبور فعلی اشتباه است.';
    }
    
    // بررسی عدم یکسانی رمز عبور جدید با قبلی
    if (!empty($current_password) && !empty($new_password) && 
        password_verify($new_password, $user['password'])) {
        $errors[] = 'رمز عبور جدید باید با رمز عبور فعلی متفاوت باشد.';
    }
    
    if (empty($errors)) {
        // هش کردن رمز عبور جدید
        $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
        
        // به‌روزرسانی رمز عبور در دیتابیس
        $stmt = $pdo->prepare("
            UPDATE users 
            SET password = :password,
                updated_at = NOW()
            WHERE id = :id
        ");
        
        $result = $stmt->execute([
            'password' => $hashed_password,
            'id' => $user_id
        ]);
        
        if ($result) {
            $message = 'رمز عبور با موفقیت تغییر یافت.';
            $success = true;
            
            // خروج از تمام دستگاه‌ها (اختیاری)
            // session_destroy();
            // header('Location: auth.php?message=password_changed');
            // exit;
        } else {
            $message = 'خطا در تغییر رمز عبور. لطفاً مجدداً تلاش کنید.';
        }
    } else {
        $message = implode('<br>', $errors);
    }
}
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="theme-color" content="#0088cc">
    <title>تغییر رمز عبور - <?= htmlspecialchars($_SESSION['first_name']) ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Vazirmatn:wght@300;400;500;600&display=swap');
        
        :root {
            --primary-color: #0088cc;
            --primary-dark: #0077b5;
            --bg-color: #f8f9fa;
            --card-bg: #ffffff;
            --text-primary: #212121;
            --text-secondary: #707579;
            --border-color: #e6e6e6;
            --error-color: #ff3b30;
            --success-color: #4cd964;
            --warning-color: #ff9500;
            --shadow: 0 4px 12px rgba(0, 0, 0, 0.08);
            --radius: 16px;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }
        
        html, body {
            height: 100%;
            width: 100%;
            font-family: 'Vazirmatn', sans-serif;
            background-color: var(--bg-color);
            color: var(--text-primary);
            font-size: 16px;
            line-height: 1.5;
        }
        
        .container {
            max-width: 500px;
            padding: 16px;
            margin: 0 auto;
            min-height: 100vh;
            display: flex;
            flex-direction: column;
        }
        
        /* هدر */
        .header {
            display: flex;
            align-items: center;
            gap: 15px;
            padding: 20px 0;
            margin-bottom: 20px;
            border-bottom: 1px solid var(--border-color);
        }
        
        .back-btn {
            background: none;
            border: none;
            color: var(--primary-color);
            font-size: 20px;
            cursor: pointer;
            padding: 10px;
            border-radius: 50%;
            transition: background-color 0.2s;
        }
        
        .back-btn:hover {
            background-color: rgba(0, 136, 204, 0.1);
        }
        
        .page-title {
            font-size: 22px;
            font-weight: 600;
            color: var(--text-primary);
        }
        
        /* کارت فرم */
        .form-card {
            background-color: var(--card-bg);
            border-radius: var(--radius);
            padding: 24px;
            box-shadow: var(--shadow);
            border: 1px solid var(--border-color);
            flex: 1;
        }
        
        /* پیام */
        .message {
            padding: 12px 16px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
            display: flex;
            align-items: center;
            gap: 10px;
            animation: slideDown 0.3s ease;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .message.error {
            background-color: rgba(255, 59, 48, 0.1);
            color: var(--error-color);
            border-right: 3px solid var(--error-color);
        }
        
        .message.success {
            background-color: rgba(76, 217, 100, 0.1);
            color: var(--success-color);
            border-right: 3px solid var(--success-color);
        }
        
        .message.warning {
            background-color: rgba(255, 149, 0, 0.1);
            color: var(--warning-color);
            border-right: 3px solid var(--warning-color);
        }
        
        .message i {
            font-size: 16px;
        }
        
        /* فیلدهای فرم */
        .input-group {
            margin-bottom: 20px;
        }
        
        .input-label {
            display: block;
            margin-bottom: 8px;
            font-size: 14px;
            color: var(--text-secondary);
            font-weight: 500;
        }
        
        .required {
            color: var(--error-color);
            margin-right: 5px;
        }
        
        .input-field {
            width: 100%;
            padding: 14px;
            border: 1.5px solid var(--border-color);
            border-radius: 10px;
            font-size: 16px;
            font-family: 'Vazirmatn', sans-serif;
            background-color: white;
            transition: border-color 0.2s;
        }
        
        .input-field:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(0, 136, 204, 0.1);
        }
        
        .input-field.error {
            border-color: var(--error-color);
        }
        
        /* قدرت رمز عبور */
        .password-strength {
            margin-top: 8px;
            height: 4px;
            background-color: var(--border-color);
            border-radius: 2px;
            overflow: hidden;
        }
        
        .password-strength-bar {
            height: 100%;
            width: 0%;
            border-radius: 2px;
            transition: width 0.3s, background-color 0.3s;
        }
        
        .strength-text {
            font-size: 12px;
            margin-top: 4px;
            text-align: left;
        }
        
        /* دکمه نمایش رمز عبور */
        .password-container {
            position: relative;
        }
        
        .toggle-password {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: var(--text-secondary);
            cursor: pointer;
            font-size: 18px;
            padding: 5px;
        }
        
        .toggle-password:hover {
            color: var(--primary-color);
        }
        
        .password-input {
            padding-left: 50px;
        }
        
        /* دکمه‌ها */
        .btn {
            padding: 14px 28px;
            border-radius: 10px;
            border: none;
            font-family: 'Vazirmatn', sans-serif;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.2s;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            text-decoration: none;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, var(--primary-color), var(--primary-dark));
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0, 136, 204, 0.3);
        }
        
        .btn-outline {
            background-color: transparent;
            color: var(--primary-color);
            border: 2px solid var(--primary-color);
        }
        
        .btn-outline:hover {
            background-color: rgba(0, 136, 204, 0.1);
        }
        
        .btn-full {
            width: 100%;
            padding: 16px;
            margin-top: 10px;
        }
        
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none !important;
            box-shadow: none !important;
        }
        
        /* راهنمای رمز عبور */
        .password-guide {
            background-color: rgba(0, 136, 204, 0.05);
            border: 1px solid rgba(0, 136, 204, 0.2);
            border-radius: 10px;
            padding: 15px;
            margin-top: 20px;
            margin-bottom: 20px;
        }
        
        .guide-title {
            color: var(--primary-color);
            font-size: 14px;
            font-weight: 600;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .guide-list {
            list-style: none;
            font-size: 13px;
            color: var(--text-secondary);
        }
        
        .guide-list li {
            margin-bottom: 6px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .guide-list li i {
            font-size: 12px;
        }
        
        /* فوتر */
        .footer {
            text-align: center;
            padding: 20px 0;
            color: var(--text-secondary);
            font-size: 14px;
            margin-top: 40px;
            border-top: 1px solid var(--border-color);
        }
        
        /* رسپانسیو */
        @media (max-width: 768px) {
            .container {
                padding: 12px;
                max-width: 100%;
            }
            
            .form-card {
                padding: 20px 16px;
                border-radius: 12px;
            }
            
            .page-title {
                font-size: 20px;
            }
            
            .btn {
                padding: 12px 24px;
                font-size: 15px;
            }
        }
        
        @media (max-width: 480px) {
            .container {
                padding: 8px;
            }
            
            .header {
                padding: 15px 0;
                margin-bottom: 15px;
            }
            
            .form-card {
                padding: 16px 12px;
                border-radius: 10px;
            }
            
            .input-field {
                padding: 12px;
                font-size: 15px;
            }
            
            .password-input {
                padding-left: 45px;
            }
            
            .toggle-password {
                left: 12px;
                font-size: 16px;
            }
            
            .btn-full {
                padding: 14px;
            }
        }
        
        /* حالت تاریک */
        @media (prefers-color-scheme: dark) {
            :root {
                --bg-color: #0f0f0f;
                --card-bg: #1a1a1a;
                --text-primary: #ffffff;
                --text-secondary: #a0a0a0;
                --border-color: #2a2a2a;
            }
            
            .input-field {
                background-color: #252525;
                color: white;
                border-color: #3a3a3a;
            }
            
            .password-guide {
                background-color: rgba(0, 136, 204, 0.1);
                border-color: rgba(0, 136, 204, 0.3);
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <!-- هدر -->
        <div class="header">
            <button class="back-btn" onclick="goBack()">
                <i class="fas fa-arrow-right"></i>
            </button>
            <h1 class="page-title">تغییر رمز عبور</h1>
        </div>
        
        <!-- کارت فرم -->
        <div class="form-card">
            <?php if ($message): ?>
                <div class="message <?= $success ? 'success' : ($success === false && $message ? 'error' : 'warning') ?>">
                    <i class="fas <?= $success ? 'fa-check-circle' : ($success === false && $message ? 'fa-exclamation-circle' : 'fa-exclamation-triangle') ?>"></i>
                    <span><?= htmlspecialchars($message) ?></span>
                </div>
            <?php endif; ?>
            
            <!-- راهنمای رمز عبور -->
            <div class="password-guide">
                <div class="guide-title">
                    <i class="fas fa-info-circle"></i>
                    <span>راهنمای انتخاب رمز عبور قوی</span>
                </div>
                <ul class="guide-list">
                    <li><i class="fas fa-check-circle" style="color: #4cd964;"></i> حداقل ۶ کاراکتر</li>
                    <li><i class="fas fa-check-circle" style="color: #4cd964;"></i> ترکیب حروف بزرگ و کوچک</li>
                    <li><i class="fas fa-check-circle" style="color: #4cd964;"></i> استفاده از اعداد و علائم</li>
                    <li><i class="fas fa-times-circle" style="color: #ff3b30;"></i> عدم استفاده از اطلاعات شخصی</li>
                </ul>
            </div>
            
            <form method="POST" id="changePasswordForm" onsubmit="return validateForm()">
                <input type="hidden" name="change_password" value="1">
                
                <div class="input-group">
                    <label class="input-label">
                        <span class="required">*</span>
                        رمز عبور فعلی
                    </label>
                    <div class="password-container">
                        <input 
                            type="password" 
                            name="current_password" 
                            class="input-field password-input" 
                            required
                            id="currentPassword"
                        >
                        <button type="button" class="toggle-password" onclick="togglePassword('currentPassword', this)">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                </div>
                
                <div class="input-group">
                    <label class="input-label">
                        <span class="required">*</span>
                        رمز عبور جدید
                    </label>
                    <div class="password-container">
                        <input 
                            type="password" 
                            name="new_password" 
                            class="input-field password-input" 
                            required
                            minlength="6"
                            id="newPassword"
                            oninput="checkPasswordStrength()"
                        >
                        <button type="button" class="toggle-password" onclick="togglePassword('newPassword', this)">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                    <div class="password-strength">
                        <div class="password-strength-bar" id="passwordStrengthBar"></div>
                    </div>
                    <div class="strength-text" id="strengthText">قدرت رمز عبور: ضعیف</div>
                </div>
                
                <div class="input-group">
                    <label class="input-label">
                        <span class="required">*</span>
                        تکرار رمز عبور جدید
                    </label>
                    <div class="password-container">
                        <input 
                            type="password" 
                            name="confirm_password" 
                            class="input-field password-input" 
                            required
                            minlength="6"
                            id="confirmPassword"
                            oninput="checkPasswordMatch()"
                        >
                        <button type="button" class="toggle-password" onclick="togglePassword('confirmPassword', this)">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                    <div class="strength-text" id="matchText"></div>
                </div>
                
                <button type="submit" class="btn btn-primary btn-full" id="submitBtn">
                    <i class="fas fa-key"></i>
                    <span>تغییر رمز عبور</span>
                </button>
                
                <a href="dashboard.php" class="btn btn-outline btn-full" style="margin-top: 10px;">
                    <i class="fas fa-times"></i>
                    <span>انصراف</span>
                </a>
            </form>
        </div>
        
        <!-- فوتر -->
        <div class="footer">
            <p>برای امنیت حساب خود، به‌طور منظم رمز عبور را تغییر دهید.</p>
        </div>
    </div>
    
    <script>
        // برگشت به صفحه قبل
        function goBack() {
            window.history.back();
        }
        
        // نمایش/مخفی کردن رمز عبور
        function togglePassword(inputId, button) {
            const input = document.getElementById(inputId);
            const icon = button.querySelector('i');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                input.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }
        
        // بررسی قدرت رمز عبور
        function checkPasswordStrength() {
            const password = document.getElementById('newPassword').value;
            const strengthBar = document.getElementById('passwordStrengthBar');
            const strengthText = document.getElementById('strengthText');
            
            let strength = 0;
            let text = 'ضعیف';
            let color = '#ff3b30';
            let width = '0%';
            
            if (password.length >= 6) strength++;
            if (password.length >= 8) strength++;
            if (password.match(/[a-z]/)) strength++;
            if (password.match(/[A-Z]/)) strength++;
            if (password.match(/[0-9]/)) strength++;
            if (password.match(/[^a-zA-Z0-9]/)) strength++;
            
            if (password.length === 0) {
                text = 'رمز عبور را وارد کنید';
                color = '#707579';
                width = '0%';
            } else if (strength <= 2) {
                text = 'ضعیف';
                color = '#ff3b30';
                width = '25%';
            } else if (strength === 3) {
                text = 'متوسط';
                color = '#ff9500';
                width = '50%';
            } else if (strength === 4) {
                text = 'قوی';
                color = '#4cd964';
                width = '75%';
            } else {
                text = 'خیلی قوی';
                color = '#4cd964';
                width = '100%';
            }
            
            strengthBar.style.width = width;
            strengthBar.style.backgroundColor = color;
            strengthText.textContent = 'قدرت رمز عبور: ' + text;
            strengthText.style.color = color;
        }
        
        // بررسی تطابق رمز عبور
        function checkPasswordMatch() {
            const password = document.getElementById('newPassword').value;
            const confirm = document.getElementById('confirmPassword').value;
            const matchText = document.getElementById('matchText');
            
            if (confirm.length === 0) {
                matchText.textContent = '';
            } else if (password === confirm) {
                matchText.textContent = '✓ رمز عبور مطابقت دارد';
                matchText.style.color = '#4cd964';
            } else {
                matchText.textContent = '✗ رمز عبور مطابقت ندارد';
                matchText.style.color = '#ff3b30';
            }
        }
        
        // اعتبارسنجی فرم
        function validateForm() {
            const form = document.getElementById('changePasswordForm');
            const submitBtn = document.getElementById('submitBtn');
            const currentPassword = document.getElementById('currentPassword').value;
            const newPassword = document.getElementById('newPassword').value;
            const confirmPassword = document.getElementById('confirmPassword').value;
            
            // اعتبارسنجی اولیه
            if (!form.checkValidity()) {
                form.reportValidity();
                return false;
            }
            
            // بررسی تطابق رمز عبور جدید
            if (newPassword !== confirmPassword) {
                alert('رمز عبور جدید و تکرار آن مطابقت ندارند.');
                return false;
            }
            
            // بررسی حداقل طول رمز عبور
            if (newPassword.length < 6) {
                alert('رمز عبور جدید باید حداقل ۶ کاراکتر باشد.');
                return false;
            }
            
            // تغییر وضعیت دکمه
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<div class="loader"></div><span>در حال تغییر رمز عبور...</span>';
            
            return true;
        }
        
        // جلوگیری از بزرگنمایی
        document.addEventListener('gesturestart', function(e) {
            e.preventDefault();
        });
        
        // مدیریت کلیدهای میانبر
        document.addEventListener('keydown', function(e) {
            // Ctrl+S یا Cmd+S برای ذخیره
            if ((e.ctrlKey || e.metaKey) && e.key === 's') {
                e.preventDefault();
                document.getElementById('changePasswordForm').submit();
            }
            
            // Escape برای برگشت
            if (e.key === 'Escape') {
                goBack();
            }
        });
        
        // بارگذاری اولیه
        document.addEventListener('DOMContentLoaded', function() {
            checkPasswordStrength();
            checkPasswordMatch();
            
            // لودر CSS
            const style = document.createElement('style');
            style.textContent = `
                .loader {
                    display: inline-block;
                    width: 18px;
                    height: 18px;
                    border: 2px solid rgba(255, 255, 255, 0.3);
                    border-top-color: white;
                    border-radius: 50%;
                    animation: spin 0.8s linear infinite;
                }
                
                @keyframes spin {
                    to { transform: rotate(360deg); }
                }
            `;
            document.head.appendChild(style);
        });
    </script>
</body>
</html>