<?php
// db_functions.php
require_once 'config.php';

/**
 * ذخیره کاربر آنلاین در دیتابیس
 */
function saveOnlineUser($user_id, $gender, $peer_id, $status = 'waiting') {
    $conn = getDBConnection();
    if (!$conn) return false;
    
    // پاکسازی کاربران منقضی شده
    $conn->query("DELETE FROM online_users WHERE expires_at < NOW()");
    
    // تولید room_id
    $room_id = 'room_' . bin2hex(random_bytes(8));
    
    // حذف کاربر قبلی
    $stmt = $conn->prepare("DELETE FROM online_users WHERE user_id = ?");
    if ($stmt) {
        $stmt->bind_param("s", $user_id);
        $stmt->execute();
        $stmt->close();
    }
    
    // ذخیره کاربر جدید
    $sql = "INSERT INTO online_users (user_id, gender, peer_id, room_id, status, expires_at) 
            VALUES (?, ?, ?, ?, ?, DATE_ADD(NOW(), INTERVAL 10 MINUTE))";
    
    $stmt = $conn->prepare($sql);
    if ($stmt) {
        $stmt->bind_param("sssss", $user_id, $gender, $peer_id, $room_id, $status);
        $result = $stmt->execute();
        $stmt->close();
        $conn->close();
        return $result ? $room_id : false;
    }
    
    $conn->close();
    return false;
}

/**
 * دریافت اطلاعات کاربر
 */
function getUserInfo($user_id) {
    $conn = getDBConnection();
    if (!$conn) return null;
    
    $sql = "SELECT * FROM online_users WHERE user_id = ? LIMIT 1";
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        $conn->close();
        return null;
    }
    
    $stmt->bind_param("s", $user_id);
    $stmt->execute();
    $stmt->store_result();
    
    if ($stmt->num_rows == 0) {
        $stmt->close();
        $conn->close();
        return null;
    }
    
    $stmt->bind_result($id, $user_id_col, $gender, $peer_id, $room_id, $status, $created_at, $updated_at, $expires_at);
    $stmt->fetch();
    
    $user = [
        'id' => $id,
        'user_id' => $user_id_col,
        'gender' => $gender,
        'peer_id' => $peer_id,
        'room_id' => $room_id,
        'status' => $status,
        'created_at' => $created_at,
        'updated_at' => $updated_at,
        'expires_at' => $expires_at
    ];
    
    $stmt->close();
    $conn->close();
    
    return $user;
}

/**
 * دریافت شریک فعلی
 */
function getCurrentPartner($user_id) {
    $conn = getDBConnection();
    if (!$conn) return null;
    
    $sql = "SELECT user1_id, user2_id, room_id FROM active_pairs 
            WHERE (user1_id = ? OR user2_id = ?) 
            AND ended_at IS NULL LIMIT 1";
    
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        $conn->close();
        return null;
    }
    
    $stmt->bind_param("ss", $user_id, $user_id);
    $stmt->execute();
    $stmt->store_result();
    
    if ($stmt->num_rows == 0) {
        $stmt->close();
        $conn->close();
        return null;
    }
    
    $stmt->bind_result($user1_id, $user2_id, $room_id);
    $stmt->fetch();
    
    $partner_id = ($user1_id == $user_id) ? $user2_id : $user1_id;
    
    $partner = [
        'partner_id' => $partner_id,
        'room_id' => $room_id
    ];
    
    $stmt->close();
    $conn->close();
    
    return $partner;
}

/**
 * آپدیت وضعیت کاربر
 */
function updateUserStatus($user_id, $status) {
    $conn = getDBConnection();
    if (!$conn) return false;
    
    $sql = "UPDATE online_users SET status = ?, updated_at = NOW(),
            expires_at = DATE_ADD(NOW(), INTERVAL 10 MINUTE) WHERE user_id = ?";
    
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        $conn->close();
        return false;
    }
    
    $stmt->bind_param("ss", $status, $user_id);
    $result = $stmt->execute();
    
    $stmt->close();
    $conn->close();
    
    return $result;
}

/**
 * یافتن شریک
 */
function findPartnerForUser($current_user_id, $current_gender) {
    $conn = getDBConnection();
    if (!$conn) return null;
    
    // پاکسازی
    $conn->query("DELETE FROM online_users WHERE expires_at < NOW()");
    
    // جستجوی شریک
    $sql = "SELECT user_id, gender, peer_id, room_id 
            FROM online_users 
            WHERE user_id != ? 
            AND status = 'waiting'
            AND expires_at > NOW()
            ORDER BY RAND()
            LIMIT 1";
    
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        $conn->close();
        return null;
    }
    
    $stmt->bind_param("s", $current_user_id);
    $stmt->execute();
    $stmt->store_result();
    
    if ($stmt->num_rows == 0) {
        $stmt->close();
        $conn->close();
        return null;
    }
    
    $stmt->bind_result($partner_id, $gender, $peer_id, $room_id);
    $stmt->fetch();
    
    $partner = [
        'user_id' => $partner_id,
        'gender' => $gender,
        'peer_id' => $peer_id,
        'room_id' => $room_id
    ];
    
    // ایجاد جفت
    if (createActivePair($current_user_id, $partner_id, $room_id)) {
        updateUserStatus($current_user_id, 'in_call');
        updateUserStatus($partner_id, 'in_call');
    } else {
        $partner = null;
    }
    
    $stmt->close();
    $conn->close();
    
    return $partner;
}

/**
 * ایجاد جفت تماس
 */
function createActivePair($user1_id, $user2_id, $room_id) {
    $conn = getDBConnection();
    if (!$conn) return false;
    
    // حذف جفت‌های قبلی
    $sql = "DELETE FROM active_pairs WHERE (user1_id = ? AND user2_id = ?) 
            OR (user1_id = ? AND user2_id = ?)";
    
    $stmt = $conn->prepare($sql);
    if ($stmt) {
        $stmt->bind_param("ssss", $user1_id, $user2_id, $user2_id, $user1_id);
        $stmt->execute();
        $stmt->close();
    }
    
    // ایجاد جفت جدید
    $sql = "INSERT INTO active_pairs (user1_id, user2_id, room_id) VALUES (?, ?, ?)";
    
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        $conn->close();
        return false;
    }
    
    $stmt->bind_param("sss", $user1_id, $user2_id, $room_id);
    $result = $stmt->execute();
    
    $stmt->close();
    $conn->close();
    
    return $result;
}

/**
 * حذف کاربر
 */
function removeOnlineUser($user_id) {
    $conn = getDBConnection();
    if (!$conn) return false;
    
    // پایان تماس‌ها
    endAllUserCalls($user_id);
    
    // حذف کاربر
    $stmt = $conn->prepare("DELETE FROM online_users WHERE user_id = ?");
    
    if (!$stmt) {
        $conn->close();
        return false;
    }
    
    $stmt->bind_param("s", $user_id);
    $result = $stmt->execute();
    
    $stmt->close();
    $conn->close();
    
    return $result;
}

/**
 * پایان تماس‌های کاربر
 */
function endAllUserCalls($user_id) {
    $conn = getDBConnection();
    if (!$conn) return;
    
    // پایان دادن به تماس‌ها
    $sql = "UPDATE active_pairs SET ended_at = NOW() 
            WHERE (user1_id = ? OR user2_id = ?) 
            AND ended_at IS NULL";
    
    $stmt = $conn->prepare($sql);
    if ($stmt) {
        $stmt->bind_param("ss", $user_id, $user_id);
        $stmt->execute();
        $stmt->close();
    }
    
    $conn->close();
}

/**
 * دریافت آمار
 */
function getStats() {
    $conn = getDBConnection();
    if (!$conn) return ['online_users' => 0, 'active_calls' => 0, 'waiting_users' => 0];
    
    $stats = [
        'online_users' => 0,
        'active_calls' => 0,
        'waiting_users' => 0
    ];
    
    // کاربران آنلاین
    $result = $conn->query("SELECT COUNT(*) as cnt FROM online_users WHERE expires_at > NOW()");
    if ($result && $row = $result->fetch_assoc()) {
        $stats['online_users'] = $row['cnt'];
    }
    
    // تماس‌های فعال
    $result = $conn->query("SELECT COUNT(*) as cnt FROM active_pairs WHERE ended_at IS NULL");
    if ($result && $row = $result->fetch_assoc()) {
        $stats['active_calls'] = $row['cnt'] * 2;
    }
    
    // کاربران منتظر
    $result = $conn->query("SELECT COUNT(*) as cnt FROM online_users WHERE status = 'waiting' AND expires_at > NOW()");
    if ($result && $row = $result->fetch_assoc()) {
        $stats['waiting_users'] = $row['cnt'];
    }
    
    $conn->close();
    
    return $stats;
}

/**
 * دریافت لیست کاربران آنلاین
 */
function getOnlineUsers($exclude_user_id = null) {
    $conn = getDBConnection();
    if (!$conn) return [];
    
    $sql = "SELECT user_id, gender, status FROM online_users WHERE expires_at > NOW()";
    
    if ($exclude_user_id) {
        $sql .= " AND user_id != ?";
    }
    
    $sql .= " ORDER BY updated_at DESC LIMIT 10";
    
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        $conn->close();
        return [];
    }
    
    if ($exclude_user_id) {
        $stmt->bind_param("s", $exclude_user_id);
    }
    
    $stmt->execute();
    $stmt->store_result();
    
    $users = [];
    
    if ($stmt->num_rows > 0) {
        $stmt->bind_result($user_id, $gender, $status);
        
        while ($stmt->fetch()) {
            $users[] = [
                'user_id' => substr($user_id, 0, 8) . '...',
                'gender' => $gender,
                'status' => $status
            ];
        }
    }
    
    $stmt->close();
    $conn->close();
    
    return $users;
}

/**
 * پاکسازی
 */
function cleanupExpiredUsers() {
    $conn = getDBConnection();
    if (!$conn) return;
    
    // پایان تماس‌های منقضی شده
    $conn->query("UPDATE active_pairs SET ended_at = NOW() 
                  WHERE ended_at IS NULL 
                  AND (user1_id IN (SELECT user_id FROM online_users WHERE expires_at < NOW())
                  OR user2_id IN (SELECT user_id FROM online_users WHERE expires_at < NOW()))");
    
    // حذف کاربران
    $conn->query("DELETE FROM online_users WHERE expires_at < NOW()");
    
    $conn->close();
}

/**
 * پایان تماس
 */
function endCall($user_id, $partner_id) {
    $conn = getDBConnection();
    if (!$conn) return false;
    
    $sql = "UPDATE active_pairs SET ended_at = NOW() 
            WHERE ((user1_id = ? AND user2_id = ?) 
            OR (user1_id = ? AND user2_id = ?)) 
            AND ended_at IS NULL";
    
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        $conn->close();
        return false;
    }
    
    $stmt->bind_param("ssss", $user_id, $partner_id, $partner_id, $user_id);
    $result = $stmt->execute();
    
    $stmt->close();
    
    if ($result) {
        updateUserStatus($user_id, 'waiting');
        updateUserStatus($partner_id, 'waiting');
    }
    
    $conn->close();
    
    return $result;
}

/**
 * ذخیره پیام چت
 */
function saveChatMessage($room_id, $user_id, $message) {
    $conn = getDBConnection();
    if (!$conn) return false;
    
    $sql = "INSERT INTO chat_messages (room_id, user_id, message) VALUES (?, ?, ?)";
    
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        $conn->close();
        return false;
    }
    
    $stmt->bind_param("sss", $room_id, $user_id, $message);
    $result = $stmt->execute();
    
    $stmt->close();
    $conn->close();
    
    return $result;
}

/**
 * دریافت پیام‌های چت
 */
function getChatMessages($room_id, $last_message_id = 0) {
    $conn = getDBConnection();
    if (!$conn) return [];
    
    $sql = "SELECT cm.id, cm.user_id, cm.message, cm.created_at, ou.gender 
            FROM chat_messages cm
            JOIN online_users ou ON cm.user_id = ou.user_id
            WHERE cm.room_id = ? AND cm.id > ?
            ORDER BY cm.created_at ASC
            LIMIT 50";
    
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        $conn->close();
        return [];
    }
    
    $stmt->bind_param("si", $room_id, $last_message_id);
    $stmt->execute();
    $stmt->store_result();
    
    $messages = [];
    
    if ($stmt->num_rows > 0) {
        $stmt->bind_result($id, $user_id, $message, $created_at, $gender);
        
        while ($stmt->fetch()) {
            $messages[] = [
                'id' => $id,
                'user_id' => $user_id,
                'message' => $message,
                'gender' => $gender,
                'created_at' => $created_at
            ];
        }
    }
    
    $stmt->close();
    $conn->close();
    
    return $messages;
}
?>