<?php
session_start();
if (isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

require_once __DIR__ . '/config/database.php';
$message = '';

// بررسی لاگین موفق
$login_success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $phone = trim($_POST['phone'] ?? '');
    $password = $_POST['password'] ?? '';

    if (empty($phone) || empty($password)) {
        $message = 'لطفاً تمام فیلدها را پر کنید.';
    } else {
        // تبدیل شماره تلفن افغانستان به فرمت بین‌المللی
        if (preg_match('/^07\d{8,9}$/', $phone)) {
            $phone = '+93' . substr($phone, 2);
        }

        // بررسی وجود کاربر در دیتابیس
        $stmt = $pdo->prepare("SELECT * FROM users WHERE phone = :phone");
        $stmt->execute(['phone' => $phone]);
        $user = $stmt->fetch();

        if ($user && password_verify($password, $user['password'])) {
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['phone'] = $user['phone'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['first_name'] = $user['first_name'];
            $_SESSION['last_name'] = $user['last_name'];
            $login_success = true;
        } else {
            $message = 'شماره تلفن یا رمز عبور اشتباه است.';
        }
    }
}
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="theme-color" content="#0088cc">
    <title>ورود - Me Chat</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" integrity="sha512-iecdLmaskl7CVkqkXNQ/ZH/XLlvWZOJyj7Yy7tcenmpD1ypASozpmT/E0iPtmFIB46ZmdtAc9eNBvH0H/ZpiBw==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <style>
        /* فونت ساده و بهینه */
        @import url('https://fonts.googleapis.com/css2?family=Vazirmatn:wght@300;400;500;600&display=swap');
        
        :root {
            --telegram-blue: #0088cc;
            --telegram-blue-dark: #0077b5;
            --telegram-green: #00a884;
            --bg-primary: #ffffff;
            --bg-secondary: #f8f9fa;
            --text-primary: #212121;
            --text-secondary: #707579;
            --border-color: #e6e6e6;
            --error-color: #ff3b30;
            --success-color: #4cd964;
        }
        
        /* ریست استایل‌ها */
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }
        
        html, body {
            height: 100%;
            width: 100%;
            overflow-x: hidden;
            font-family: 'Vazirmatn', 'Segoe UI', Tahoma, sans-serif;
            background-color: var(--bg-primary);
            color: var(--text-primary);
            line-height: 1.5;
            font-size: 16px;
        }
        
        /* حذف تمام انیمیشن‌های سنگین */
        /* فقط از transition ساده استفاده می‌کنیم */
        
        .container {
            min-height: 100vh;
            display: flex;
            flex-direction: column;
            position: relative;
        }
        
        /* هدر شبیه تلگرام */
        .header {
            background-color: var(--telegram-blue);
            color: white;
            padding: 60px 20px 40px;
            text-align: center;
            position: relative;
            border-bottom-left-radius: 20px;
            border-bottom-right-radius: 20px;
            box-shadow: 0 2px 10px rgba(0, 136, 204, 0.2);
        }
        
        .logo {
            margin-bottom: 20px;
        }
        
        .logo-icon {
            width: 80px;
            height: 80px;
            background-color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 15px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }
        
        .logo-icon i {
            font-size: 40px;
            color: var(--telegram-blue);
        }
        
        .logo-text {
            font-size: 28px;
            font-weight: 600;
            letter-spacing: -0.5px;
        }
        
        .header p {
            font-size: 15px;
            opacity: 0.9;
            margin-top: 5px;
        }
        
        /* بخش فرم */
        .form-container {
            flex: 1;
            padding: 30px 20px;
            position: relative;
            z-index: 1;
            margin-top: -20px;
            background-color: var(--bg-primary);
            border-radius: 20px 20px 0 0;
        }
        
        .form-box {
            max-width: 400px;
            margin: 0 auto;
        }
        
        /* پیام خطا */
        .message {
            background-color: rgba(255, 59, 48, 0.1);
            color: var(--error-color);
            padding: 12px 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
            border-right: 3px solid var(--error-color);
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .message i {
            font-size: 18px;
        }
        
        /* گروه فیلدها */
        .input-group {
            margin-bottom: 20px;
        }
        
        .input-label {
            display: block;
            margin-bottom: 8px;
            font-size: 14px;
            color: var(--text-secondary);
            font-weight: 500;
        }
        
        .input-field {
            width: 100%;
            padding: 15px;
            border: 2px solid var(--border-color);
            border-radius: 12px;
            font-size: 16px;
            font-family: 'Vazirmatn', sans-serif;
            background-color: white;
            transition: border-color 0.2s;
            -webkit-appearance: none;
            appearance: none;
        }
        
        .input-field:focus {
            outline: none;
            border-color: var(--telegram-blue);
        }
        
        /* فیلد شماره تلفن */
        .phone-input-container {
            position: relative;
        }
        
        .phone-prefix {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            background-color: var(--bg-secondary);
            padding: 8px 12px;
            border-radius: 8px;
            font-weight: 600;
            color: var(--text-primary);
            display: flex;
            align-items: center;
            gap: 8px;
            border: 1px solid var(--border-color);
        }
        
        .phone-input {
            padding-left: 100px;
            direction: ltr;
            text-align: left;
        }
        
        /* دکمه */
        .btn {
            width: 100%;
            padding: 16px;
            background-color: var(--telegram-blue);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 16px;
            font-weight: 600;
            font-family: 'Vazirmatn', sans-serif;
            cursor: pointer;
            transition: background-color 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            margin-top: 10px;
        }
        
        .btn:hover {
            background-color: var(--telegram-blue-dark);
        }
        
        .btn:active {
            transform: scale(0.98);
        }
        
        .btn i {
            font-size: 18px;
        }
        
        /* لینک ثبت‌نام */
        .register-link {
            text-align: center;
            margin-top: 30px;
            padding-top: 20px;
            border-top: 1px solid var(--border-color);
        }
        
        .register-link a {
            color: var(--telegram-blue);
            text-decoration: none;
            font-weight: 600;
            display: inline-block;
            padding: 5px 10px;
            border-radius: 8px;
            transition: background-color 0.2s;
        }
        
        .register-link a:hover {
            background-color: rgba(0, 136, 204, 0.1);
        }
        
        /* لودر ساده */
        .loader {
            width: 24px;
            height: 24px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-top-color: white;
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* استایل‌های موبایل */
        @media (max-width: 480px) {
            .header {
                padding: 40px 15px 30px;
            }
            
            .logo-icon {
                width: 70px;
                height: 70px;
            }
            
            .logo-icon i {
                font-size: 35px;
            }
            
            .logo-text {
                font-size: 24px;
            }
            
            .form-container {
                padding: 25px 15px;
            }
            
            .input-field {
                padding: 14px;
                font-size: 15px;
            }
            
            .btn {
                padding: 15px;
                font-size: 15px;
            }
            
            .phone-prefix {
                padding: 7px 10px;
                font-size: 14px;
            }
            
            .phone-input {
                padding-left: 90px;
            }
        }
        
        /* استایل‌های تبلت */
        @media (min-width: 768px) {
            .container {
                max-width: 500px;
                margin: 0 auto;
                border: 1px solid var(--border-color);
                border-radius: 20px;
                overflow: hidden;
                min-height: auto;
                margin-top: 40px;
                box-shadow: 0 5px 25px rgba(0, 0, 0, 0.1);
            }
            
            .header {
                border-radius: 0;
            }
            
            .form-container {
                margin-top: 0;
                border-radius: 0;
            }
        }
        
        /* جلوگیری از زوم روی موبایل */
        input, textarea, select {
            font-size: 16px; /* از بزرگنمایی خودکار در iOS جلوگیری می‌کند */
        }
        
        /* بهینه‌سازی برای دستگاه‌های لمسی */
        @media (hover: none) and (pointer: coarse) {
            .btn:active {
                background-color: var(--telegram-blue-dark);
                transform: scale(0.95);
            }
            
            .input-field {
                font-size: 16px; /* از بزرگنمایی جلوگیری می‌کند */
            }
        }
        
        /* حالت آفلاین */
        .offline-indicator {
            position: fixed;
            top: 0;
            right: 0;
            left: 0;
            background-color: #ff9500;
            color: white;
            text-align: center;
            padding: 10px;
            font-size: 14px;
            z-index: 1000;
            display: none;
        }
        
        /* حالت تاریک (اختیاری) */
        @media (prefers-color-scheme: dark) {
            :root {
                --bg-primary: #0f0f0f;
                --bg-secondary: #1a1a1a;
                --text-primary: #ffffff;
                --text-secondary: #a0a0a0;
                --border-color: #2a2a2a;
            }
            
            .input-field {
                background-color: var(--bg-secondary);
                color: var(--text-primary);
                border-color: var(--border-color);
            }
            
            .phone-prefix {
                background-color: var(--bg-primary);
                border-color: var(--border-color);
            }
        }
    </style>
</head>
<body>
    <div class="offline-indicator" id="offlineIndicator">
        <i class="fas fa-wifi-slash"></i> اتصال به اینترنت ضعیف است
    </div>
    
    <div class="container">
        <div class="header">
            <div class="logo">
                <div class="logo-icon">
                    <i class="fas fa-paper-plane"></i>
                </div>
                <div class="logo-text">Me Chat</div>
                <p>پیام‌رسان سریع و امن</p>
            </div>
        </div>
        
        <div class="form-container">
            <div class="form-box">
                <?php if ($message): ?>
                    <div class="message">
                        <i class="fas fa-exclamation-circle"></i>
                        <span><?= htmlspecialchars($message) ?></span>
                    </div>
                <?php endif; ?>
                
                <form method="POST" id="loginForm">
                    <div class="input-group">
                        <label class="input-label">شماره تلفن</label>
                        <div class="phone-input-container">
                            <input 
                                type="tel" 
                                name="phone" 
                                class="input-field phone-input" 
                                placeholder="7XXXXXXXX" 
                                required 
                                value="<?= htmlspecialchars($_POST['phone'] ?? '') ?>"
                                inputmode="numeric"
                                pattern="07[0-9]{8,9}"
                                title="شماره تلفن باید با 07 شروع شود و 10-11 رقم باشد"
                            >
                            <div class="phone-prefix">
                                <span>+93</span>
                                <i class="fas fa-flag"></i>
                            </div>
                        </div>
                    </div>
                    
                    <div class="input-group">
                        <label class="input-label">رمز عبور</label>
                        <input 
                            type="password" 
                            name="password" 
                            class="input-field" 
                            placeholder="رمز عبور خود را وارد کنید" 
                            required
                            minlength="6"
                        >
                    </div>
                    
                    <button type="submit" class="btn" id="loginButton">
                        <span>ورود به حساب</span>
                        <i class="fas fa-sign-in-alt"></i>
                    </button>
                </form>
                
                <div class="register-link">
                    <span>حساب کاربری ندارید؟</span>
                    <a href="register.php">ثبت‌نام کنید</a>
                </div>
            </div>
        </div>
    </div>

    <script>
        // بهینه‌سازی برای نت ضعیف
        document.addEventListener('DOMContentLoaded', function() {
            const loginForm = document.getElementById('loginForm');
            const loginButton = document.getElementById('loginButton');
            const offlineIndicator = document.getElementById('offlineIndicator');
            
            // بررسی وضعیت آنلاین/آفلاین
            function updateOnlineStatus() {
                if (navigator.onLine) {
                    offlineIndicator.style.display = 'none';
                } else {
                    offlineIndicator.style.display = 'block';
                }
            }
            
            window.addEventListener('online', updateOnlineStatus);
            window.addEventListener('offline', updateOnlineStatus);
            updateOnlineStatus(); // وضعیت اولیه
            
            // مدیریت فرم با تایم‌اوت برای نت ضعیف
            if (loginForm) {
                loginForm.addEventListener('submit', function(e) {
                    if (this.checkValidity()) {
                        e.preventDefault();
                        
                        // غیرفعال کردن دکمه
                        const originalText = loginButton.innerHTML;
                        loginButton.disabled = true;
                        loginButton.innerHTML = `
                            <div class="loader"></div>
                            <span>در حال اتصال...</span>
                        `;
                        
                        // تایم‌اوت برای نت ضعیف
                        const submitTimeout = setTimeout(() => {
                            // اگر بعد از 10 ثانیه پاسخی نیامد
                            if (loginButton.disabled) {
                                loginButton.disabled = false;
                                loginButton.innerHTML = originalText;
                                alert('اتصال اینترنت ضعیف است. لطفاً مجدداً تلاش کنید.');
                            }
                        }, 10000);
                        
                        // ارسال فرم با fetch برای کنترل بهتر
                        const formData = new FormData(this);
                        
                        fetch(window.location.href, {
                            method: 'POST',
                            body: formData,
                            headers: {
                                'X-Requested-With': 'XMLHttpRequest'
                            }
                        })
                        .then(response => {
                            clearTimeout(submitTimeout);
                            return response.text();
                        })
                        .then(html => {
                            // بررسی اگر پاسخ شامل پیام خطا بود
                            if (html.includes('شماره تلفن یا رمز عبور اشتباه') || 
                                html.includes('لطفاً تمام فیلدها را پر کنید')) {
                                // نمایش پیام خطا بدون ریلود کامل صفحه
                                const tempDiv = document.createElement('div');
                                tempDiv.innerHTML = html;
                                const errorMsg = tempDiv.querySelector('.message');
                                
                                if (errorMsg) {
                                    const existingMsg = document.querySelector('.message');
                                    if (existingMsg) {
                                        existingMsg.remove();
                                    }
                                    loginForm.insertBefore(errorMsg, loginForm.firstChild);
                                }
                                
                                loginButton.disabled = false;
                                loginButton.innerHTML = originalText;
                            } else {
                                // اگر لاگین موفق بود، ریدایرکت
                                window.location.href = 'index1.php';
                            }
                        })
                        .catch(error => {
                            clearTimeout(submitTimeout);
                            console.error('Error:', error);
                            loginButton.disabled = false;
                            loginButton.innerHTML = originalText;
                            alert('خطا در ارتباط با سرور. لطفاً دوباره تلاش کنید.');
                        });
                    }
                });
            }
            
            // اعتبارسنجی فوری شماره تلفن
            const phoneInput = document.querySelector('input[name="phone"]');
            if (phoneInput) {
                phoneInput.addEventListener('input', function() {
                    const value = this.value.replace(/\D/g, '');
                    if (value.startsWith('0')) {
                        this.value = value;
                    } else if (value.length > 0) {
                        this.value = '0' + value;
                    }
                    
                    // اعتبارسنجی بصری
                    if (this.value.length >= 10 && this.value.length <= 11 && this.value.startsWith('07')) {
                        this.style.borderColor = 'var(--success-color)';
                    } else if (this.value.length > 0) {
                        this.style.borderColor = 'var(--error-color)';
                    } else {
                        this.style.borderColor = 'var(--border-color)';
                    }
                });
            }
            
            // جلوگیری از ارسال فرم با Enter اگر فرم معتبر نیست
            loginForm.addEventListener('keydown', function(e) {
                if (e.key === 'Enter') {
                    if (!this.checkValidity()) {
                        e.preventDefault();
                        // اسکرول به اولین فیلد نامعتبر
                        const firstInvalid = this.querySelector(':invalid');
                        if (firstInvalid) {
                            firstInvalid.focus();
                            firstInvalid.scrollIntoView({ behavior: 'smooth', block: 'center' });
                        }
                    }
                }
            });
            
            // کش کردن فونت‌ها و آیکون‌ها
            if ('serviceWorker' in navigator) {
                window.addEventListener('load', function() {
                    navigator.serviceWorker.register('/sw.js').catch(function(error) {
                        console.log('ServiceWorker registration failed:', error);
                    });
                });
            }
            
            // جلوگیری از بزرگنمایی با دو انگشت
            document.addEventListener('gesturestart', function(e) {
                e.preventDefault();
            });
            
            // بهینه‌سازی برای CPU ضعیف
            let lastScrollTop = 0;
            window.addEventListener('scroll', function() {
                const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
                // فقط هر 100px اسکرول هندل شود
                if (Math.abs(scrollTop - lastScrollTop) > 100) {
                    lastScrollTop = scrollTop;
                }
            }, { passive: true });
        });
        
        // فالنک برای زمانی که جاوااسکریپت غیرفعال است
        document.getElementById('loginForm').addEventListener('submit', function() {
            if (this.checkValidity()) {
                this.querySelector('button').disabled = true;
            }
        });
    </script>
</body>
</html>