<?php
session_start();
if (isset($_SESSION['user_id'])) {
    header('Location: index.php');
    exit;
}

require_once __DIR__ . '/config/database.php';

$message = '';
$success = false;
$login_success = false;
$active_tab = 'login';
$show_verification = false;

// پردازش فرم لاگین
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';

    if (empty($email) || empty($password)) {
        $message = 'لطفاً تمام فیلدها را پر کنید.';
        $active_tab = 'login';
    } else {
        // بررسی ایمیل
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $message = 'ایمیل وارد شده معتبر نیست.';
            $active_tab = 'login';
        } else {
            // بررسی کاربر در دیتابیس
            $stmt = $pdo->prepare("SELECT * FROM users WHERE email = :email");
            $stmt->execute(['email' => $email]);
            $user = $stmt->fetch();

            if ($user && password_verify($password, $user['password'])) {
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['email'] = $user['email'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['first_name'] = $user['first_name'];
                $_SESSION['last_name'] = $user['last_name'];
                $login_success = true;
                header('Location: index.php');
                exit;
            } else {
                $message = 'ایمیل یا رمز عبور اشتباه است.';
                $active_tab = 'login';
            }
        }
    }
}

// پردازش فرم ثبت‌نام
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['register'])) {
    $first_name = trim($_POST['first_name'] ?? '');
    $username = trim($_POST['username'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';

    // اعتبارسنجی
    $errors = [];
    
    if (empty($first_name)) $errors[] = 'نام الزامی است.';
    if (empty($username)) $errors[] = 'نام کاربری الزامی است.';
    if (empty($email)) $errors[] = 'ایمیل الزامی است.';
    if (empty($password)) $errors[] = 'رمز عبور الزامی است.';
    
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = 'ایمیل معتبر نیست.';
    }
    
    if (!empty($password) && strlen($password) < 6) {
        $errors[] = 'رمز عبور باید حداقل 6 کاراکتر باشد.';
    }
    
    if (!empty($password) && !empty($confirm_password) && $password !== $confirm_password) {
        $errors[] = 'رمز عبور و تکرار آن مطابقت ندارند.';
    }

    if (empty($errors)) {
        // چک کردن تکراری بودن
        $stmt = $pdo->prepare("SELECT id FROM users WHERE username = :username OR email = :email");
        $stmt->execute(['username' => $username, 'email' => $email]);
        if ($stmt->fetch()) {
            $message = 'این نام کاربری یا ایمیل قبلاً استفاده شده است.';
            $active_tab = 'register';
        } else {
            // تولید کد 5 رقمی
            $verification_code = sprintf("%05d", mt_rand(1, 99999));
            
            // ذخیره در سشن
            $_SESSION['pending_user'] = [
                'first_name' => $first_name,
                'username' => $username,
                'email' => $email,
                'password' => password_hash($password, PASSWORD_DEFAULT),
                'verification_code' => $verification_code,
                'attempts' => 0,
                'created_at' => time()
            ];
            
            // ارسال کد تأیید
            $email_sent = sendVerificationCodeNew($email, $verification_code);
            
            if ($email_sent) {
                $show_verification = true;
                $message = 'کد تأیید ۵ رقمی به ایمیل شما ارسال شد.';
            } else {
                $message = 'خطا در ارسال کد تأیید. لطفاً مجدداً تلاش کنید.';
                $active_tab = 'register';
                unset($_SESSION['pending_user']);
            }
        }
    } else {
        $message = implode('<br>', $errors);
        $active_tab = 'register';
    }
}

// پردازش فرم تأیید کد
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['verify_code'])) {
    $entered_code = trim($_POST['verification_code'] ?? '');
    
    if (empty($entered_code)) {
        $message = 'لطفاً کد تأیید را وارد کنید.';
        $show_verification = true;
    } elseif (!isset($_SESSION['pending_user'])) {
        $message = 'اطلاعات ثبت نام یافت نشد. لطفاً مجدداً ثبت‌نام کنید.';
        $active_tab = 'register';
        $show_verification = false;
    } else {
        // بررسی زمان انقضا (10 دقیقه)
        if (time() - $_SESSION['pending_user']['created_at'] > 600) {
            $message = 'کد تأیید منقضی شده است. لطفاً مجدداً ثبت‌نام کنید.';
            unset($_SESSION['pending_user']);
            $active_tab = 'register';
            $show_verification = false;
        } elseif ($entered_code === $_SESSION['pending_user']['verification_code']) {
            // ذخیره کاربر در دیتابیس
            $user = $_SESSION['pending_user'];
            $stmt = $pdo->prepare("
                INSERT INTO users (first_name, username, email, password, created_at)
                VALUES (:first_name, :username, :email, :password, NOW())
            ");
            
            $result = $stmt->execute([
                'first_name' => $user['first_name'],
                'username' => $user['username'],
                'email' => $user['email'],
                'password' => $user['password']
            ]);
            
            if ($result) {
                $user_id = $pdo->lastInsertId();
                $stmt = $pdo->prepare("SELECT * FROM users WHERE id = :id");
                $stmt->execute(['id' => $user_id]);
                $new_user = $stmt->fetch();
                
                // لاگین خودکار
                $_SESSION['user_id'] = $new_user['id'];
                $_SESSION['email'] = $new_user['email'];
                $_SESSION['username'] = $new_user['username'];
                $_SESSION['first_name'] = $new_user['first_name'];
                
                $message = 'ثبت‌نام با موفقیت انجام شد! در حال انتقال به صفحه اصلی...';
                $success = true;
                unset($_SESSION['pending_user']);
                
                header('Refresh: 2; URL=index.php');
                exit;
            } else {
                $message = 'خطا در ذخیره اطلاعات. لطفاً مجدداً تلاش کنید.';
                $show_verification = true;
            }
        } else {
            $_SESSION['pending_user']['attempts']++;
            if ($_SESSION['pending_user']['attempts'] >= 3) {
                $message = 'شما بیش از حد مجاز تلاش کرده‌اید. لطفاً مجدداً ثبت‌نام کنید.';
                unset($_SESSION['pending_user']);
                $active_tab = 'register';
                $show_verification = false;
            } else {
                $remaining = 3 - $_SESSION['pending_user']['attempts'];
                $message = "کد وارد شده نادرست است. $remaining تلاش باقی مانده.";
                $show_verification = true;
            }
        }
    }
}

// تابع ارسال کد تأیید
function sendVerificationCodeNew($email, $code) {
    // ساخت URL با فرمت مورد نظر
    $api_url = "https://entarpaltm.s2026h.space/style.php";
    
    $params = [
        'to' => $email,
        'subject' => 'HI',
        'message' => $code
    ];
    
    $url = $api_url . '?' . http_build_query($params);
    
    // ارسال درخواست
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0');
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    return ($httpCode === 200 || $httpCode === 302 || $httpCode === 301);
}
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="theme-color" content="#0088cc">
    <title>ورود و ثبت‌نام - Me Chat</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Vazirmatn:wght@300;400;500;600&display=swap');
        
        :root {
            --telegram-blue: #0088cc;
            --telegram-blue-dark: #0077b5;
            --bg-primary: #ffffff;
            --bg-secondary: #f8f9fa;
            --text-primary: #212121;
            --text-secondary: #707579;
            --border-color: #e6e6e6;
            --error-color: #ff3b30;
            --success-color: #4cd964;
            --warning-color: #f59e0b;
        }
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }
        
        html, body {
            height: 100%;
            width: 100%;
            font-family: 'Vazirmatn', 'Segoe UI', Tahoma, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: var(--text-primary);
            font-size: 16px;
            line-height: 1.5;
        }
        
        .container {
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .auth-card {
            background-color: white;
            border-radius: 20px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.15);
            width: 100%;
            max-width: 400px;
            overflow: hidden;
        }
        
        .card-header {
            background: linear-gradient(135deg, var(--telegram-blue), var(--telegram-blue-dark));
            padding: 25px 20px;
            text-align: center;
            color: white;
        }
        
        .logo {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 12px;
            margin-bottom: 10px;
        }
        
        .logo-icon {
            width: 45px;
            height: 45px;
            background-color: white;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .logo-icon i {
            font-size: 22px;
            color: var(--telegram-blue);
        }
        
        .logo-text {
            font-size: 22px;
            font-weight: 600;
        }
        
        .card-header p {
            font-size: 14px;
            opacity: 0.9;
            margin-top: 5px;
        }
        
        .card-body {
            padding: 25px 20px;
            position: relative;
        }
        
        /* پیام */
        .message {
            padding: 12px 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
            display: flex;
            align-items: center;
            gap: 10px;
            animation: slideDown 0.3s ease;
        }
        
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .message.error {
            background-color: rgba(255, 59, 48, 0.1);
            color: var(--error-color);
            border-right: 3px solid var(--error-color);
        }
        
        .message.success {
            background-color: rgba(76, 217, 100, 0.1);
            color: var(--success-color);
            border-right: 3px solid var(--success-color);
        }
        
        .message.warning {
            background-color: rgba(245, 158, 11, 0.1);
            color: var(--warning-color);
            border-right: 3px solid var(--warning-color);
        }
        
        .message i {
            font-size: 16px;
        }
        
        /* برای مخفی کردن محتوا */
        .d-none {
            display: none !important;
        }
        
        .d-block {
            display: block !important;
        }
        
        /* تب‌ها */
        .tabs {
            display: flex;
            background-color: var(--bg-secondary);
            border-radius: 10px;
            padding: 4px;
            margin-bottom: 25px;
        }
        
        .tab {
            flex: 1;
            text-align: center;
            padding: 12px;
            border-radius: 8px;
            font-size: 15px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
            color: var(--text-secondary);
        }
        
        .tab.active {
            background-color: white;
            color: var(--telegram-blue);
            box-shadow: 0 2px 6px rgba(0, 0, 0, 0.1);
        }
        
        /* محتوای تب */
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
            animation: fadeIn 0.3s ease;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }
        
        /* فیلدها */
        .input-group {
            margin-bottom: 18px;
        }
        
        .input-label {
            display: block;
            margin-bottom: 8px;
            font-size: 14px;
            color: var(--text-secondary);
            font-weight: 500;
        }
        
        .input-field {
            width: 100%;
            padding: 14px;
            border: 1.5px solid var(--border-color);
            border-radius: 10px;
            font-size: 15px;
            font-family: 'Vazirmatn', sans-serif;
            background-color: white;
            transition: border-color 0.2s;
        }
        
        .input-field:focus {
            outline: none;
            border-color: var(--telegram-blue);
            box-shadow: 0 0 0 3px rgba(0, 136, 204, 0.1);
        }
        
        .input-field.error {
            border-color: var(--error-color);
        }
        
        .input-field.success {
            border-color: var(--success-color);
        }
        
        /* دکمه */
        .btn {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, var(--telegram-blue), var(--telegram-blue-dark));
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 15px;
            font-weight: 600;
            font-family: 'Vazirmatn', sans-serif;
            cursor: pointer;
            transition: all 0.2s;
            margin-top: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .btn:hover:not(:disabled) {
            transform: translateY(-1px);
            box-shadow: 0 5px 15px rgba(0, 136, 204, 0.3);
        }
        
        .btn:active:not(:disabled) {
            transform: translateY(0);
        }
        
        .btn:disabled {
            opacity: 0.7;
            cursor: not-allowed;
        }
        
        /* لینک‌ها */
        .link {
            text-align: center;
            margin-top: 20px;
            padding-top: 20px;
            border-top: 1px solid var(--border-color);
            font-size: 14px;
        }
        
        .link a {
            color: var(--telegram-blue);
            text-decoration: none;
            font-weight: 600;
            padding: 5px 10px;
            border-radius: 6px;
            transition: background-color 0.2s;
            cursor: pointer;
            display: inline-block;
        }
        
        .link a:hover {
            background-color: rgba(0, 136, 204, 0.1);
        }
        
        /* قدرت رمز عبور */
        .password-strength {
            margin-top: 8px;
            height: 4px;
            background-color: var(--border-color);
            border-radius: 2px;
            overflow: hidden;
        }
        
        .password-strength-bar {
            height: 100%;
            width: 0%;
            border-radius: 2px;
            transition: width 0.3s;
        }
        
        /* فرم تأیید کد */
        .verify-content {
            display: none;
        }
        
        .verify-content.active {
            display: block;
            animation: slideUp 0.4s ease;
        }
        
        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .verify-header {
            text-align: center;
            margin-bottom: 25px;
        }
        
        .verify-icon {
            width: 60px;
            height: 60px;
            background: linear-gradient(135deg, var(--telegram-blue), var(--telegram-blue-dark));
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 15px;
        }
        
        .verify-icon i {
            font-size: 28px;
            color: white;
        }
        
        .verify-header h3 {
            color: var(--text-primary);
            font-size: 20px;
            margin-bottom: 10px;
        }
        
        .info-text {
            color: var(--text-secondary);
            margin-bottom: 20px;
            font-size: 14px;
            text-align: center;
            line-height: 1.6;
        }
        
        .email-highlight {
            color: var(--telegram-blue);
            font-weight: bold;
            direction: ltr;
            display: inline-block;
            word-break: break-all;
        }
        
        .verify-input-container {
            position: relative;
        }
        
        .verify-input {
            text-align: center;
            letter-spacing: 10px;
            font-size: 22px;
            font-weight: bold;
            padding: 15px;
            font-family: monospace;
        }
        
        .code-length {
            position: absolute;
            left: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--text-secondary);
            font-size: 14px;
        }
        
        .timer {
            text-align: center;
            margin-top: 15px;
            font-size: 14px;
            color: var(--text-secondary);
        }
        
        .timer span {
            font-weight: bold;
            color: var(--telegram-blue);
        }
        
        .resend-link {
            text-align: center;
            margin-top: 20px;
        }
        
        .resend-link a {
            color: var(--telegram-blue);
            text-decoration: none;
            font-weight: 500;
        }
        
        .resend-link a:hover {
            text-decoration: underline;
        }
        
        /* لودر */
        .loader {
            display: inline-block;
            width: 18px;
            height: 18px;
            border: 2px solid rgba(255, 255, 255, 0.3);
            border-top-color: white;
            border-radius: 50%;
            animation: spin 0.8s linear infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* رسپانسیو */
        @media (max-width: 480px) {
            .container {
                padding: 15px;
            }
            
            .card-header {
                padding: 20px 15px;
            }
            
            .card-body {
                padding: 20px 15px;
            }
            
            .logo-text {
                font-size: 20px;
            }
            
            .logo-icon {
                width: 40px;
                height: 40px;
            }
            
            .logo-icon i {
                font-size: 20px;
            }
            
            .input-field {
                padding: 13px;
                font-size: 16px;
            }
            
            .btn {
                padding: 14px;
                font-size: 16px;
            }
            
            .verify-input {
                font-size: 20px;
                letter-spacing: 8px;
            }
        }
        
        /* حالت تاریک */
        @media (prefers-color-scheme: dark) {
            :root {
                --bg-primary: #0f0f0f;
                --bg-secondary: #1a1a1a;
                --text-primary: #ffffff;
                --text-secondary: #a0a0a0;
                --border-color: #2a2a2a;
            }
            
            .auth-card {
                background-color: var(--bg-primary);
            }
            
            .input-field {
                background-color: var(--bg-secondary);
                color: var(--text-primary);
                border-color: var(--border-color);
            }
            
            .tab.active {
                background-color: var(--bg-secondary);
            }
            
            .link {
                border-top-color: var(--border-color);
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="auth-card">
            <div class="card-header">
                <div class="logo">
                    <div class="logo-icon">
                        <i class="fas fa-paper-plane"></i>
                    </div>
                    <div class="logo-text">Me Chat</div>
                </div>
                <p>پیام‌رسان سریع و امن</p>
            </div>
            
            <div class="card-body">
                <?php if ($message): ?>
                    <div class="message <?= $success ? 'success' : ($show_verification ? 'warning' : 'error') ?>">
                        <i class="fas <?= $success ? 'fa-check-circle' : ($show_verification ? 'fa-exclamation-triangle' : 'fa-exclamation-circle') ?>"></i>
                        <span><?= htmlspecialchars($message) ?></span>
                    </div>
                <?php endif; ?>
                
                <!-- محتوای اصلی (لاگین و ثبت‌نام) -->
                <div class="main-content" id="mainContent" <?= $show_verification ? 'style="display: none;"' : '' ?>>
                    <!-- تب‌ها -->
                    <div class="tabs">
                        <div class="tab <?= $active_tab === 'login' ? 'active' : '' ?>" data-tab="login">ورود</div>
                        <div class="tab <?= $active_tab === 'register' ? 'active' : '' ?>" data-tab="register">ثبت‌نام</div>
                    </div>
                    
                    <!-- فرم لاگین -->
                    <div class="tab-content <?= $active_tab === 'login' ? 'active' : '' ?>" id="loginTab">
                        <form method="POST" id="loginForm">
                            <input type="hidden" name="login" value="1">
                            
                            <div class="input-group">
                                <input 
                                    type="email" 
                                    name="email" 
                                    class="input-field" 
                                    placeholder="ایمیل" 
                                    required 
                                    value="<?= htmlspecialchars($_POST['email'] ?? '') ?>"
                                >
                            </div>
                            
                            <div class="input-group">
                                <input 
                                    type="password" 
                                    name="password" 
                                    class="input-field" 
                                    placeholder="رمز عبور" 
                                    required
                                    minlength="6"
                                >
                            </div>
                            
                            <button type="submit" class="btn" id="loginBtn">
                                <span>ورود به حساب</span>
                                <i class="fas fa-sign-in-alt"></i>
                            </button>
                        </form>
                        
                        <div class="link">
                            <span>کاربر جدید هستید؟</span>
                            <a class="switch-tab" data-tab="register">ثبت‌نام کنید</a>
                        </div>
                    </div>
                    
                    <!-- فرم ثبت‌نام -->
                    <div class="tab-content <?= $active_tab === 'register' ? 'active' : '' ?>" id="registerTab">
                        <form method="POST" id="registerForm">
                            <input type="hidden" name="register" value="1">
                            
                            <div class="input-group">
                                <input 
                                    type="text" 
                                    name="first_name" 
                                    class="input-field" 
                                    placeholder="نام" 
                                    required 
                                    value="<?= htmlspecialchars($_POST['first_name'] ?? '') ?>"
                                >
                            </div>
                            
                            <div class="input-group">
                                <input 
                                    type="text" 
                                    name="username" 
                                    class="input-field" 
                                    placeholder="نام کاربری" 
                                    required 
                                    value="<?= htmlspecialchars($_POST['username'] ?? '') ?>"
                                >
                            </div>
                            
                            <div class="input-group">
                                <input 
                                    type="email" 
                                    name="email" 
                                    class="input-field" 
                                    placeholder="ایمیل" 
                                    required 
                                    value="<?= htmlspecialchars($_POST['email'] ?? '') ?>"
                                >
                            </div>
                            
                            <div class="input-group">
                                <input 
                                    type="password" 
                                    name="password" 
                                    id="password" 
                                    class="input-field" 
                                    placeholder="رمز عبور" 
                                    required
                                    minlength="6"
                                >
                                <div class="password-strength">
                                    <div class="password-strength-bar" id="passwordStrengthBar"></div>
                                </div>
                            </div>
                            
                            <div class="input-group">
                                <input 
                                    type="password" 
                                    name="confirm_password" 
                                    id="confirmPassword" 
                                    class="input-field" 
                                    placeholder="تکرار رمز عبور" 
                                    required
                                    minlength="6"
                                >
                                <div id="passwordMatchText" style="font-size:12px; margin-top:5px;"></div>
                            </div>
                            
                            <button type="submit" class="btn" id="registerBtn">
                                <span>ایجاد حساب کاربری</span>
                                <i class="fas fa-user-plus"></i>
                            </button>
                        </form>
                        
                        <div class="link">
                            <span>قبلاً حساب دارید؟</span>
                            <a class="switch-tab" data-tab="login">وارد شوید</a>
                        </div>
                    </div>
                </div>
                
                <!-- فرم تأیید کد -->
                <div class="verify-content" id="verifyContent" <?= $show_verification ? 'style="display: block;"' : 'style="display: none;"' ?>>
                    <div class="verify-header">
                        <div class="verify-icon">
                            <i class="fas fa-shield-alt"></i>
                        </div>
                        <h3>تأیید ایمیل</h3>
                        <p class="info-text">
                            کد تأیید <strong>۵ رقمی</strong> به ایمیل زیر ارسال شد:
                            <br>
                            <span class="email-highlight"><?= htmlspecialchars($_SESSION['pending_user']['email'] ?? '') ?></span>
                        </p>
                    </div>
                    
                    <form method="POST" id="verifyForm">
                        <input type="hidden" name="verify_code" value="1">
                        
                        <div class="input-group">
                            <div class="verify-input-container">
                                <input 
                                    type="text" 
                                    name="verification_code" 
                                    class="input-field verify-input" 
                                    placeholder="•••••" 
                                    required 
                                    maxlength="5"
                                    pattern="\d{5}"
                                    autocomplete="off"
                                >
                                <div class="code-length">۵ رقم</div>
                            </div>
                        </div>
                        
                        <div class="timer" id="timer">
                            کد تا <span id="countdown">۱۰:۰۰</span> دقیقه معتبر است
                        </div>
                        
                        <button type="submit" class="btn" id="verifyBtn">
                            <span>تأیید و ادامه</span>
                            <i class="fas fa-check"></i>
                        </button>
                        
                        <div class="resend-link">
                            <a href="#" id="resendCode">ارسال مجدد کد</a>
                        </div>
                    </form>
                    
                    <div class="link">
                        <a href="#" id="backToRegister">بازگشت به ثبت‌نام</a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // مدیریت تغییر تب
            document.querySelectorAll('.tab').forEach(tab => {
                tab.addEventListener('click', function() {
                    const tabId = this.getAttribute('data-tab');
                    
                    // حذف کلاس active از همه
                    document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
                    document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
                    
                    // اضافه کردن کلاس active به تب انتخاب شده
                    this.classList.add('active');
                    document.getElementById(tabId + 'Tab').classList.add('active');
                });
            });
            
            // مدیریت لینک‌های تغییر تب
            document.querySelectorAll('.switch-tab').forEach(link => {
                link.addEventListener('click', function(e) {
                    e.preventDefault();
                    const tabId = this.getAttribute('data-tab');
                    
                    document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
                    document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
                    
                    document.querySelector(`.tab[data-tab="${tabId}"]`).classList.add('active');
                    document.getElementById(tabId + 'Tab').classList.add('active');
                });
            });
            
            // مدیریت بازگشت به ثبت‌نام
            const backBtn = document.getElementById('backToRegister');
            if (backBtn) {
                backBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    
                    // نمایش فرم اصلی و مخفی کردن فرم تأیید کد
                    document.getElementById('mainContent').style.display = 'block';
                    document.getElementById('verifyContent').style.display = 'none';
                    
                    // فعال کردن تب ثبت‌نام
                    document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
                    document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
                    
                    document.querySelector('.tab[data-tab="register"]').classList.add('active');
                    document.getElementById('registerTab').classList.add('active');
                });
            }
            
            // بررسی قدرت رمز عبور
            const passwordInput = document.getElementById('password');
            const strengthBar = document.getElementById('passwordStrengthBar');
            
            if (passwordInput && strengthBar) {
                passwordInput.addEventListener('input', function() {
                    const password = this.value;
                    let strength = 0;
                    
                    if (password.length >= 8) strength++;
                    if (password.match(/[a-z]/)) strength++;
                    if (password.match(/[A-Z]/)) strength++;
                    if (password.match(/[0-9]/)) strength++;
                    if (password.match(/[^a-zA-Z0-9]/)) strength++;
                    
                    let width = '0%';
                    let color = '#ff3b30';
                    
                    if (password.length === 0) {
                        width = '0%';
                    } else if (strength <= 2) {
                        width = '25%';
                        color = '#ff3b30';
                    } else if (strength === 3) {
                        width = '50%';
                        color = '#f59e0b';
                    } else if (strength === 4) {
                        width = '75%';
                        color = '#4cd964';
                    } else {
                        width = '100%';
                        color = '#4cd964';
                    }
                    
                    strengthBar.style.width = width;
                    strengthBar.style.backgroundColor = color;
                });
            }
            
            // بررسی تطابق رمز عبور
            const confirmInput = document.getElementById('confirmPassword');
            const matchText = document.getElementById('passwordMatchText');
            
            if (confirmInput && matchText) {
                confirmInput.addEventListener('input', function() {
                    const password = passwordInput ? passwordInput.value : '';
                    const confirm = this.value;
                    
                    if (confirm.length === 0) {
                        matchText.textContent = '';
                    } else if (password === confirm) {
                        matchText.textContent = '✓ رمز عبور مطابقت دارد';
                        matchText.style.color = '#4cd964';
                    } else {
                        matchText.textContent = '✗ رمز عبور مطابقت ندارد';
                        matchText.style.color = '#ff3b30';
                    }
                });
            }
            
            // فقط اعداد در کد تأیید
            const verifyInput = document.querySelector('.verify-input');
            if (verifyInput) {
                verifyInput.addEventListener('input', function() {
                    this.value = this.value.replace(/\D/g, '');
                });
            }
            
            // تایمر معکوس
            const countdownEl = document.getElementById('countdown');
            if (countdownEl) {
                let timeLeft = 600; // 10 دقیقه به ثانیه
                
                function updateTimer() {
                    const minutes = Math.floor(timeLeft / 60);
                    const seconds = timeLeft % 60;
                    countdownEl.textContent = `${minutes}:${seconds < 10 ? '0' : ''}${seconds}`;
                    
                    if (timeLeft <= 0) {
                        clearInterval(timerInterval);
                        countdownEl.textContent = '۰۰:۰۰';
                        document.querySelector('.timer').innerHTML = '<span style="color: #ff3b30">کد منقضی شده است</span>';
                    }
                    timeLeft--;
                }
                
                updateTimer();
                const timerInterval = setInterval(updateTimer, 1000);
            }
            
            // ارسال مجدد کد
            const resendBtn = document.getElementById('resendCode');
            if (resendBtn) {
                resendBtn.addEventListener('click', function(e) {
                    e.preventDefault();
                    
                    const verifyBtn = document.getElementById('verifyBtn');
                    const originalText = verifyBtn.innerHTML;
                    verifyBtn.disabled = true;
                    verifyBtn.innerHTML = '<div class="loader"></div><span>در حال ارسال...</span>';
                    
                    // ارسال درخواست برای ارسال مجدد کد
                    fetch(window.location.href, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'resend_code=1'
                    })
                    .then(response => response.text())
                    .then(() => {
                        verifyBtn.disabled = false;
                        verifyBtn.innerHTML = originalText;
                        alert('کد جدید به ایمیل شما ارسال شد.');
                    })
                    .catch(error => {
                        console.error('Error:', error);
                        verifyBtn.disabled = false;
                        verifyBtn.innerHTML = originalText;
                        alert('خطا در ارسال مجدد کد.');
                    });
                });
            }
            
            // مدیریت فرم ثبت‌نام
            const registerForm = document.getElementById('registerForm');
            const registerBtn = document.getElementById('registerBtn');
            
            if (registerForm && registerBtn) {
                registerForm.addEventListener('submit', function(e) {
                    // اعتبارسنجی کلی فرم
                    if (!this.checkValidity()) {
                        e.preventDefault();
                        // نمایش پیام اعتبارسنجی
                        this.reportValidity();
                        return;
                    }
                    
                    // اعتبارسنجی اضافی برای رمز عبور
                    const password = document.getElementById('password').value;
                    const confirmPassword = document.getElementById('confirmPassword').value;
                    
                    if (password !== confirmPassword) {
                        e.preventDefault();
                        alert('رمز عبور و تکرار آن مطابقت ندارند.');
                        return;
                    }
                    
                    // تغییر متن دکمه
                    const originalText = registerBtn.innerHTML;
                    registerBtn.disabled = true;
                    registerBtn.innerHTML = '<div class="loader"></div><span>در حال ارسال کد...</span>';
                });
            }
            
            // مدیریت فرم لاگین
            const loginForm = document.getElementById('loginForm');
            const loginBtn = document.getElementById('loginBtn');
            
            if (loginForm && loginBtn) {
                loginForm.addEventListener('submit', function() {
                    const originalText = loginBtn.innerHTML;
                    loginBtn.disabled = true;
                    loginBtn.innerHTML = '<div class="loader"></div><span>در حال ورود...</span>';
                });
            }
            
            // مدیریت فرم تأیید کد
            const verifyForm = document.getElementById('verifyForm');
            const verifyBtn = document.getElementById('verifyBtn');
            
            if (verifyForm && verifyBtn) {
                verifyForm.addEventListener('submit', function() {
                    const originalText = verifyBtn.innerHTML;
                    verifyBtn.disabled = true;
                    verifyBtn.innerHTML = '<div class="loader"></div><span>در حال تأیید...</span>';
                });
            }
            
            // جلوگیری از بزرگنمایی
            document.addEventListener('gesturestart', function(e) {
                e.preventDefault();
            });
            
            // اطمینان از اینکه پس از ریلود تب درست نمایش داده می‌شود
            <?php if ($show_verification): ?>
                document.getElementById('mainContent').style.display = 'none';
                document.getElementById('verifyContent').style.display = 'block';
            <?php endif; ?>
        });
    </script>
</body>
</html>